/*
 *  SPECjEnterprise2010 - a benchmark for enterprise middleware
 *  Copyright 1995-2010 Standard Performance Evaluation Corporation
 *  All Rights Reserved
 *
 * This source code is provided as is, without any express or implied warranty.
 *
 *  History:
 *  Date        ID, Company             Description
 *  ----------  ----------------        ----------------------------------------------
 *  2003/01/01  John Stecher, IBM       Created for SPECjAppServer2004
 *  2003/04/24  John Stecher, IBM       Updated doPurchase to return a ArrayList because we
 *                                      are now using the session bean to return the ArrayList
 *                                      as a wrapper around the items, and the orderDataBean
 *  2003/05/01  John Stecher, IBM       Imported Russ Raymundo's mfg side of the benchmark for web base driving
 *  2003/05/06  John Stecher, IBM       Made changes to allow drive instead of application to determine
 *                                      if an order is to be deferred or added to inventory immediately
 *  2003/08/30  John Stecher, IBM       Updated for new sell functionality
 *  2003/11/25  Tom Daly    , Sun       Added category to getVehicleQuotes();
 *  2003/12/05  John Stecher, IBM       Added Atomicity Tests
 *  2003/01/08  John Stecher, IBM       Changed code to eliminate unused objects being passed into methods
 *  2004/01/25  John Stecher, IBM       Added code for cache consistancy test.
 *  2008/04/08  Pinaki Poddar, BEA      1. Modified Exception behavior of all methods
 *                                      2. getItems() methods return List<Item>
 *                                      instead of a Map. This list has the
 *                                      same information as the Map had. 
 *                                      3. Methods are more type-strict than
 *                                      simply trying to manipulate String/int
 *                                      id as instance reference. This is now
 *                                      possible because the instances are POJO
 *                                      according to JPA. 
 *                                      4. ShoppingCart now captures the 
 *                                      information on active session including
 *                                      the Customer. Hence methods now take
 *                                      only a ShoppingCart and not a Customer. 
 *                                      5. Primitive Wrapper as identifier in
 *                                      method argument is replaced by their
 *                                      non-wrapped primitive. Saves null pointer. 
 * 2010/05/08  Anoop Gupta, Oracle      Added location parameter to operations on WorkOrder as 
 *                                      WorkOrder should be looked up at specific location
 * 2010/08/09 Tom Daly, Sun             Added getItemJDBC and refactored getItem to getItemJPA
 *                                      
 */

package org.spec.jent.servlet.helper;

import java.math.BigDecimal;
import java.sql.SQLException;
import java.util.Calendar;
import java.util.Collection;
import java.util.List;

import javax.naming.NamingException;
import javax.servlet.http.HttpSession;

import org.spec.jent.ejb.mfg.entity.WorkOrder;
import org.spec.jent.ejb.orders.entity.Customer;
import org.spec.jent.ejb.orders.entity.CustomerInventory;
import org.spec.jent.ejb.orders.entity.Item;
import org.spec.jent.ejb.orders.entity.Order;
import org.spec.jent.ejb.orders.session.InsufficientCreditException;
import org.spec.jent.ejb.shoppingcart.entity.ShoppingCart;


/**
  * SpecServices interface specifies the business methods provided by the SpecJ
  * online Dealer application.
  * These business methods represent the features and operations that can be 
  * performed by dealers such as login, logout, view the shopping cart, 
  * buy or sell a vehicle, etc.
  *
  */


public interface SpecServices {

    /**
     * Gets the Item given its identifier.
     * 
     */
    public Item getItemJPA(String itemID);

    
    /**
     * Changes price of an item given by the identifier via JDBC
     * 
     */
    public void updateItemJDBC(String itemID, BigDecimal newPrice) throws SQLException, NamingException;

    /**
      * Gets all open orders for a given customer.
      *
      */
    public List<Order> getOpenOrders(int userID);

    /**
      * Gets a list of the current vehicles either specified in the search or
      * instead just the generic search of all vehicle.  
      * Displays them 10 at a time.
      * 
      * @param vehicles list of all vehicles to search for in the 
      * manufacturer's inventory.
      * @param session - the http session that is used to hold a reference to 
      * the state-full session bean containing the inventory
      * @param browse - an indicator to the application on which direction the 
      * dealer want to browse the inventory
      * @return the VehicleSearch - a wrapper object containing all the vehicle 
      * to display on the current page
      */
    public VehicleSearch getVehicleQuotes(List<String> vehicles, 
            HttpSession session, String browse, int category);

    /**
      * Return the portfolio of stock holdings for the specified customer
      * as a collection of HoldingDataBeans
      *
      * @param userID the customer requesting the portfolio     
      * @return Collection of the users portfolio of stock holdings
      */
    public Collection<CustomerInventory> getHoldings(int userID);    

    /**
     * Cancel an open order and remove the order from the dealer's list of
     * orders to be processed in the future.
     *
     * @return boolean indicating if canceled the order without error
     */
    public boolean cancel(int orderID) throws Exception;    

    /**
     * Sell a vehicle and removed the vehicle and quantity for the given dealer.
     * Given a vehicle, remove the inventory, credit the dealerships account,
     * and remove the vehicles from the dealership's portfolio.
     *
     * @param userID the customer requesting the sell
     * @param holdingID the vehicle to be sold
     * @return boolean indicating if the sale occurred without error
     */
    public boolean sell(int userID, long holdingID, boolean isAtomicityTest);

    /**
     * Return an Customer for the given userID.
     *
     * @param userID the dealer's userID to lookup
     */
    public Customer getCustomerData(int userID);
    
    /**
     * Purchase the items currently in the shopping cart.
     *
     * @param userID the customer who wishes to purchase
     * @param sc the shopping cart that contains the items the dealership wishes to purchase
     * @return ArrayList containing the orderDataBean and the items HashMap
     */
    public Order doPurchase(int customerId, ShoppingCart cart, int location, boolean deferred,
            boolean rollback) throws InsufficientCreditException;

    /**
     * Attempt to authenticate and login a user
     *
     */
    public boolean login(int userID);                              

    /**
     * Logout the given user
     *
     * @param userID the customer to logout 
     * @return the login status
     */
    public boolean logout(HttpSession session); 

    /**
     * Schedule A Work Order
     *
     * @param assemblyId Assembly ID
     * @param qty Quantity
     * @param location WorkOrder processing Location
     * @param dueDate Due Date
     * @return status of operation
     */
    public WorkOrder scheduleWorkOrder(String assemblyId, int qty, 
            int location, Calendar dueDate);

    /**
     * Schedule A Work Order
     *
     * @param salesId       Sales order id
     * @param oLineId       Order Line ID
     * @param assemblyId    Assembly Id
     * @param qty           Original Qty
     * @param dueDate       Date when order is due
     * @return woID         Work Order ID
     */
    public WorkOrder scheduleWorkOrder(int salesId, int oLineId, 
            String assemblyId, int qty, int location, Calendar dueDate);

    /**
     * Get Work Order Status
     *
     * @param woID the work order to cancel
     * @param location the location of work order
     * @return status of WorkOrder
     */
    public String getWorkOrderStatus(int woID, int location);

    /**
     * Cancel A Work Order
     *
     * @param woID the work order to cancel
     * @param location the location of work order
     * @return status of operation
     */
    public boolean cancelWorkOrder(int woID, int location);

    /**
     * Complete A Work Order
     *
     * @param woID the work order to Complete
     * @param location the location of work order
     * @return status of operation
     */
    public boolean completeWorkOrder(int woID, int location);

    /**
     * Update A Work Order
     *
     * @param woID the work order to update
     * @param location the location of work order
     * @return status of operation
     */
    public boolean updateWorkOrder(int woID, int location);

    /**
    * Get All Assembly
    *
    * @return List of Assembly IDs
    */
    public List<String> getAssemblyIds();
    
    /**
     * This test checks to see if the atomic nature of a transaction is 
     * being upheld by the benchmark application.  
     * This test case places an order for immediate insertion into the dealer's 
     * inventory.  An exception is raised after placing the order and while 
     * adding the inventory.  This exception should roll back  
     * the transaction and all other 
     * items returned to how they existed before the transaction took place.
     * This test case has three steps which are as follows to verify atomicity
     * 
     * 1.)  Query database to check how many inventory items the dealership has, 
     * the dealer's account balance, and the number of orders which have been 
     * placed for the dealer inside of the dealer domain.  These number are the 
     * initial metrics that the final test cases should line up with after 
     * rolling back the transaction.
     * 
     * 2.)  Drives the above listed transaction which causes a transaction 
     * roll back exception to occur
     * 
     * 3.)  Query database to check how many inventory items the dealer has, 
     * the dealer's account balance, and the number of orders which have been 
     * placed for the dealer inside of the dealer domain.  
     * These number should equal those in step 1) for the test case to be 
     * successful and the transaction to have been atomic.
     * 
     * @return if the test passed or failed
     * @throws Exception
     */
    public boolean atomicityTestOne() throws Exception;
    
    /**
     * This test transaction simply tests that the application server is working 
     * properly and that it is able to insert an order as in Atomicity
     * test 1 but without causing the exception and have it show up in the database.
     * 
     * @return if the test passed or failed
     * @throws Exception
     */
    public boolean atomicityTestTwo() throws Exception;

    /**
     * This test checks to see if the proper transaction atomicity levels are 
     * being upheld in transaction associated with the benchmark
     * and specifically the messaging subsystem in this test case.  This test 
     * case drives placing a order which contains a large order and
     * an item to be insert immediately into the dealer inventory.  An exception 
     * is raised after placing the order and while adding the inventory to the 
     * dealers inventory 
     * table.  This should cause the transaction changes to be removed from the 
     * database, messages removed from queue and all other items returned to how 
     * they existed 
     * before the transaction took place.  This test case has three steps which 
     * are as follows to verify atomicity
     *  
     * 1.)  Query database to check how many inventory items the dealer has, the 
     * dealer account balance, and the number of 
     * orders which have been placed for the dealer inside of the dealer domain.  
     * Also the large order table is queried to check how 
     * many large orders exist in the database before we drive the transaction.  
     * These number are the initial metrics that the final
     * test cases should line up with after rolling back the transaction.
     * 
     * 2.)  Drives the above listed transaction which causes a transaction roll 
     * back exception to occur
     * 
     * 3.)  Query database to check how many inventory items the dealer has, the
     * dealer account balance, and the number of 
     * orders which have been placed for the dealer inside of the dealer domain.
     * Also query the large order table to check how many
     * large orders there are in the table.  These number should equal those in 
     * step 1) for
     * the test case to be successful and the transaction to have been atomic.
     * 
     * @return if the test passed or failed
     * @throws Exception
     */ 
    public boolean atomicityTestThree() throws Exception;
    
}   

