/*
 *   SPECjEnterprise2010 - a benchmark for enterprise middleware
 *  Copyright 1995-2010 Standard Performance Evaluation Corporation
 *   All Rights Reserved
 *
 * This source code is provided as is, without any express or implied warranty.
 *
 *  History:
 *  Date        ID, Company               Description
 *  ----------  ----------------          ----------------------------------------------
 *  2003/01/01  John Stecher, IBM         Created for SPECjAppServer2004
 *  2003/04/15	John Stecher, IBM         added debugging
 *  2003/04/20	John Stecher, IBM         Explicitly remove session beans after use
 *  2003/04/24	John Stecher, IBM         made changes to handle the correct usage of the 
 *                                        orderSes session bean actually returning a object
 *                                        wrapper containing the orderDataBean and ItemsDataBean
 *  2003/05/01  John Stecher, IBM         Imported Russ Raymundo's mfg side of the benchmark for web base driving
 *  2003/05/05	John Stecher, IBM         Made changes to catch noSuchObject exception on remove and display 
 *                                        warning message.
 *  2003/05/05	John Stecher, IBM         Made changes to ensure that users are updated when car is sold out from 
 *                                        under them with a warning message.
 *  2003/05/06	John Stecher, IBM         Made changes to allow drive instead of application to determine
 *                                        if an order is to be deferred or added to inventory immediately
 *  2003/06/24  John Stecher, IBM         Made changes to conform with best practices guidelines
 *  2003/06/28  John Stecher, IBM         Updated BigDecimal usage for better performance
 *  2003/08/30  John Stecher, IBM         Updated for new sell functionality
 *  2003/11/25  Tom Daly    , Sun         added support for category in doVehicleQuotes()
 *  2003/12/05  John Stecher, IBM         Added Atomicity Tests
 *  2003/01/08  John Stecher, IBM         Changed code to eliminate unused objects being passed into methods
 *  2004/01/25  John Stecher, IBM         Added code for cache consistancy test.
 *  2004/02/27  Samuel Kounev, Darmstadt  Changed to call printStackTrace when catching Exceptions.
 *  2004/03/11  John Stecher, IBM         Updated to make cache test work correctly in a cluster
 *  
 *  2008/04/08  Pinaki Poddar BEA         Refactored error processing
 *  2009/05/08  Anoop Gupta, Oracle       Added location parameter to operations on WorkOrder as 
 *                                        WorkOrder should be looked up at specific location
 *  2009/08/09 Tom Daly, Sun              Modified cache test to check updated item price via JDBC instead of JPA
 */

package org.spec.jent.servlet.helper;

import static org.spec.jent.servlet.helper.SpecConfig.ATOMICITY_PAGE;
import static org.spec.jent.servlet.helper.SpecConfig.CACHETEST_PAGE;
import static org.spec.jent.servlet.helper.SpecConfig.ERROR_PAGE;
import static org.spec.jent.servlet.helper.SpecConfig.HOME_PAGE;
import static org.spec.jent.servlet.helper.SpecConfig.INVENTORY_PAGE;
import static org.spec.jent.servlet.helper.SpecConfig.MFG_HOME_PAGE;
import static org.spec.jent.servlet.helper.SpecConfig.MFG_SCHEDULEWO_PAGE;
import static org.spec.jent.servlet.helper.SpecConfig.MFG_WELCOME_PAGE;
import static org.spec.jent.servlet.helper.SpecConfig.MFG_WORKORDER_CHECK_PAGE;
import static org.spec.jent.servlet.helper.SpecConfig.MFG_WORKORDER_PAGE;
import static org.spec.jent.servlet.helper.SpecConfig.ORDER_PAGE;
import static org.spec.jent.servlet.helper.SpecConfig.SHOPPINGCART_PAGE;
import static org.spec.jent.servlet.helper.SpecConfig.SUPPLIER_HOME_PAGE;
import static org.spec.jent.servlet.helper.SpecConfig.VEHICLEQUOTE_PAGE;
import static org.spec.jent.servlet.helper.SpecConfig.WELCOME_PAGE;
import static org.spec.jent.servlet.helper.SpecConfig.getPage;

import java.io.IOException;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Collection;
import java.util.List;

import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.spec.jent.ejb.mfg.entity.WorkOrder;
import org.spec.jent.ejb.orders.entity.Customer;
import org.spec.jent.ejb.orders.entity.CustomerInventory;
import org.spec.jent.ejb.orders.entity.Item;
import org.spec.jent.ejb.orders.entity.Order;
import org.spec.jent.ejb.orders.session.InsufficientCreditException;
import org.spec.jent.ejb.shoppingcart.entity.ShoppingCart;
import org.spec.jent.ejb.supplier.entity.Supplier;

/**
 * SpecServletAction provides servlet specific client side access to each of the
 * Spec vehicle dealership user operations. These include login, logout, buy, 
 * sell, browse, etc.
 * SpecServletAction manages a web interface to the SPECjEnterprise application 
 * handling HttpRequests/HttpResponse objects and forwarding results to the 
 * appropriate JSP page for the web interface.
 * SpecServletAction invokes {@link SpecAction} methods to actually perform
 * each dealership operation.
 *
 */
public class SpecServletAction {
    // List of keys used to put data in the HttpSession
    // Declaring them statically 
    //       saves lots of String construction 
    //       reduces chance of programming typos error.
    // 
    public static final String KEY_CUSTOMER  = "customer";
    public static final String KEY_CUSTOMER_DATA = "customerData";
    public static final String KEY_CART      = "shoppingcart";
    public static final String KEY_INVENTORY = "inventory";
    public static final String KEY_ITEMS     = "itemInfo";
    public static final String KEY_LOCATION  = "location";
    public static final String KEY_ORDER     = "order";
    public static final String KEY_ORDERS    = "openOrders";
    public static final String KEY_SUPPLIERS = "suppliers";
    public static final String KEY_RESULT    = "results";
    public static final String KEY_ERROR     = "error";
    public static final String KEY_MESSAGE   = "message";
    public static final String KEY_STATUS    = "status";
    public static final String KEY_EXCEPTION = "exception";
    public static final String KEY_INVENTORY_PAGE_VISIT = "inventoryPageVisits";
    public static final String KEY_CART_PAGE_VISIT = "shoppingCartPageVisits";
    public static final String KEY_BROWSER_PAGE_VISIT = "browsePageVisits";
    public static final String KEY_HOME_PAGE_VISIT = "homepageVisits";
    public static final String KEY_ORDER_CANCELLED = "ordersCancelled";
    public static final String KEY_ORDER_PLACED = "ordersPlaced";
    public static final String KEY_INVENTORY_SOLD = "holdingsSold";
    public static final String KEY_SALE_PROFIT = "totalSalesProfits";
    public static final String KEY_PURCHASE_DEBITS = "totalPurchaseDebits";
    public static final String KEY_CUSTOMER_ID = "uidBean";
    public static final String KEY_CREATE_DATE = "sessionCreationDate";
    public static final String KEY_CATEGORY = "category";
    
    private SpecAction sAction = new SpecAction();
    
    public SpecServletAction() {
    	SpecBeanFactory.initalize();
    }
    
    /**
     * Login a Spec dealership User.
     * Dispatch to the SpecJHome JSP for display
     * 
     * @param userID The User to login
     * @param ctx the servlet context
     * @param req the HttpRequest object
     * @param resp the HttpResponse object
     * @exception ServletException if a servlet specific exception occurs
     * @exception IOException if fails to write respose back to the user
     * 
     */
    public void doLogin(ServletContext ctx, 
                       HttpServletRequest req, 
                       HttpServletResponse resp, 
                       Integer userID) 
    throws ServletException, java.io.IOException {
        try {
            boolean loggedIn = sAction.login(userID);
            if( loggedIn ) {
                HttpSession session = req.getSession(false);
                if( session != null) {
                    session.invalidate(); 
                }
                session = req.getSession(true);
                session.setAttribute(KEY_CUSTOMER_ID, userID);
                session.setAttribute(KEY_CREATE_DATE, new java.util.Date());
                session.setAttribute(KEY_INVENTORY_PAGE_VISIT, 0);
                session.setAttribute(KEY_CART_PAGE_VISIT, 0);
                session.setAttribute(KEY_BROWSER_PAGE_VISIT, 0);
                session.setAttribute(KEY_HOME_PAGE_VISIT, 0);
                session.setAttribute(KEY_ORDER_CANCELLED, 0);
                session.setAttribute(KEY_ORDER_PLACED, 0);
                session.setAttribute(KEY_INVENTORY_SOLD,0);
                session.setAttribute(KEY_SALE_PROFIT, SpecUtils.ZERO);
                session.setAttribute(KEY_PURCHASE_DEBITS, SpecUtils.ZERO);
                doHome(ctx, req, resp, userID, "Ready to work");
            } else {
                req.setAttribute(KEY_RESULT, "Could not find account " + userID);
                requestDispatch(ctx, req, resp, getPage(WELCOME_PAGE)); 
            }
        } catch( Exception ex) {
            processError(ctx, req, resp, "doLogin", 1, ex);
        }
    }
   
    void incrementVisit(HttpSession session, String keyPage) {
        Object visits = session.getAttribute(keyPage);
        int increment = (visits == null) ? 1 : ((Integer)visits).intValue()+1;
        session.setAttribute(keyPage, increment);
    }

    void processError(ServletContext ctx, HttpServletRequest req, 
            HttpServletResponse resp, String msg, int status, Exception ex) 
    throws IOException, ServletException {
        req.setAttribute(KEY_ERROR, "internal error");
        req.setAttribute(KEY_MESSAGE, msg);
        req.setAttribute(KEY_STATUS, "1");
        req.setAttribute(KEY_EXCEPTION, ex);
        requestDispatch(ctx, req, resp, getPage(ERROR_PAGE));
    }

    /**
     * Display Dealership inventory information such as closed orders, open 
     * orders, and overall number of vehicles currently in the inventory.  
     * Dispatch to the Spec dealerinventory JSP for display
     *
     * @param userID The User to display profile info
     * @param ctx the servlet context
     * @param req the HttpRequest object
     * @param resp the HttpResponse object
     * @param results A short description of the results/success of this web request provided on the web page
     * @exception ServletException if a servlet specific exception occurs
     * @exception IOException if fails to write respose back to the user
     * 
     */
    public void doInventory(ServletContext ctx, 
                           HttpServletRequest req, 
                           HttpServletResponse resp, 
                           Integer userID)
    throws ServletException, java.io.IOException {
        try {
            Collection<CustomerInventory>  customerDataBeans = sAction.getHoldings(userID);        
            List<Order>      orderDataBean   = sAction.getOpenOrders(userID);
            
            req.setAttribute(KEY_INVENTORY, customerDataBeans);
            req.setAttribute(KEY_ORDERS, orderDataBean);
            req.setAttribute(KEY_RESULT, "");
            
            HttpSession session = req.getSession(true);
            incrementVisit(session, KEY_INVENTORY_PAGE_VISIT);
            requestDispatch(ctx, req, resp, getPage(INVENTORY_PAGE));
        } catch( Exception ex) {
            processError(ctx, req, resp, "doInventory()", 1, ex);
        }
    }

    /**
     * Cancel a non-completed order for the given dealership
     * Dispatch to the Spec dealerinventory JSP for display 
     *
     * @param userID The dealership cancelling its order
     * @param ctx the servlet context
     * @param req the HttpRequest object
     * @param resp the HttpResponse object
     * @exception ServletException if a servlet specific exception occurs
     * @exception IOException if fails to write respose back to the user
     * 
     */
    public void doCancelOrder(ServletContext ctx, 
                             HttpServletRequest req, 
                             HttpServletResponse resp,
                             Integer userID)
    throws ServletException, IOException {
        try {
            int orderID = Integer.parseInt(req.getParameter("orderID"));
            boolean cancelled = sAction.cancel(orderID);
            HttpSession session = req.getSession(true);
            incrementVisit(session, KEY_ORDER_CANCELLED);
            if(!cancelled){
            	req.setAttribute("results_cancel", "*** NOTICE: The order you tried to cancel has already been cancelled! ***");	
            }	
            this.doInventory(ctx, req, resp, userID);
        } catch( Exception ex) {
            processError(ctx, req, resp, "doCancelOrder()", 1, ex);
        }
    } 

    /**
     * Create the Spec Home page with personalized information such as the 
     * dealerships account balance.
     * Dispatch to the SpecJHome JSP for display
     *
     * @param ctx the servlet context
     * @param req the HttpRequest object
     * @param resp the HttpResponse object
     * @param results A short description of the results/success of this web request provided on the web page
     * @exception ServletException if a servlet specific exception occurs
     * @exception IOException if fails to write respose back to the user
     * 
     */
    public void doHome(ServletContext ctx, 
                      HttpServletRequest req, 
                      HttpServletResponse resp, 
                      Integer userID, 
                      String results)
    throws ServletException, java.io.IOException {
        try {
            HttpSession session = req.getSession();          
            Customer customer = sAction.getCustomerData(userID);
            Collection<CustomerInventory>  inventories = sAction.getHoldings(userID);
            req.setAttribute(KEY_CUSTOMER_DATA, customer);            
            req.setAttribute(KEY_INVENTORY, inventories);            
            req.setAttribute(KEY_RESULT, results);
            incrementVisit(session, KEY_HOME_PAGE_VISIT);
            requestDispatch(ctx, req, resp, getPage(HOME_PAGE));
        } catch( Exception ex) {
            processError(ctx, req, resp, "doHome()", 1, ex);
        }
    }

    /**
     * Logout a Spec dealership User
     * Dispatch to the Spec Welcome JSP for display
     * 
     * @param userID The User to logout
     * @param ctx the servlet context
     * @param req the HttpRequest object
     * @param resp the HttpResponse object
     * @exception ServletException if a servlet specific exception occurs
     * @exception IOException if fails to write respose back to the user
     * 
     */
    public void doLogout(ServletContext ctx, 
                        HttpServletRequest req, 
                        HttpServletResponse resp, 
                        Integer userID)
    throws ServletException, IOException {
        HttpSession session = req.getSession();
        try {
            sAction.logout(session);
        } catch( Exception ex) {
            processError(ctx, req, resp, "doLogout()", 1, ex);
        }
        if( session != null ) {
            session.invalidate();
        }
        req.setAttribute(KEY_RESULT, "");
        requestDispatch(ctx, req, resp, getPage(WELCOME_PAGE));
    }

    /**
     * Retrieve the current dealerships shopping cart of vehicles
     * Dispatch to the Spec shoppingcart JSP for display 
     *
     * @param userID The User requesting to view their dealership shopping cart
     * @param ctx the servlet context
     * @param req the HttpRequest object
     * @param resp the HttpResponse object
     * @exception ServletException if a servlet specific exception occurs
     * @exception IOException if fails to write respose back to the user
     * 
     */
    public void doShoppingCart(ServletContext ctx, 
                              HttpServletRequest req, 
                              HttpServletResponse resp,
                              Integer userID)
    throws ServletException, IOException {
        HttpSession session = req.getSession();     
        ShoppingCart cart = getShoppingCart(session);
        req.setAttribute(KEY_CART, cart);
        incrementVisit(session, KEY_CART_PAGE_VISIT);        
        requestDispatch(ctx, req, resp, getPage(SHOPPINGCART_PAGE)); 
    }

    /**
     * Add an item to the current dealerships shopping cart of vehicles
     * Dispatch to the Spec shoppingcart JSP for display 
     *
     * @param userID The User requesting to add to their dealership's shopping cart
     * @param ctx the servlet context
     * @param req the HttpRequest object
     * @param resp the HttpResponse object
     * @exception ServletException if a servlet specific exception occurs
     * @exception IOException if fails to write respose back to the user
     * 
     */   
    public void doAddToShoppingCart(ServletContext ctx, 
                                   HttpServletRequest req, 
                                   HttpServletResponse resp,
                                   Integer userID)
    throws ServletException, IOException {
        HttpSession session = req.getSession();
        ShoppingCart cart = getShoppingCart(session);
        Item item = sAction.getItemJPA(req.getParameter(KEY_ITEM_ID));
        cart.addItem(item, Integer.parseInt(req.getParameter(KEY_QUANTITY)));
        req.setAttribute(KEY_CART, cart);
        incrementVisit(session, KEY_CART_PAGE_VISIT);
        requestDispatch(ctx, req, resp, getPage(SHOPPINGCART_PAGE)); 
    }

    public static final String KEY_ITEM_ID = "itemId";
    public static final String KEY_QUANTITY = "quantity";
    
    /**
     * Remove an item from the current dealerships shopping cart of vehicles
     * Dispatch to the Spec shoppingcart JSP for display 
     *
     * @param userID The User requesting to remove from their dealership's shopping cart
     * @param ctx the servlet context
     * @param req the HttpRequest object
     * @param resp the HttpResponse object
     * @exception ServletException if a servlet specific exception occurs
     * @exception IOException if fails to write respose back to the user
     * 
     */    
    public void doRemoveFromShoppingCart(ServletContext ctx, 
                                        HttpServletRequest req, 
                                        HttpServletResponse resp,
                                        Integer userID)
    throws ServletException, IOException {
        HttpSession session = req.getSession();
        ShoppingCart cart = getShoppingCart(session);
        String removeID = req.getParameter("cartID");
        cart.removeItem(removeID);
        postInvoke(req, cart);
        requestDispatch(ctx, req, resp, getPage(SHOPPINGCART_PAGE)); 
    }

    void postInvoke(HttpServletRequest req, ShoppingCart cart) {
        HttpSession session = req.getSession();
        req.setAttribute(KEY_CART, cart);
        incrementVisit(session, KEY_CART_PAGE_VISIT);
    }

    /**
     * Clear all items from the current dealerships shopping cart of vehicles
     * Dispatch to the Spec shoppingcart JSP for display 
     *
     * @param userID The User requesting to clear their dealership's shopping cart
     * @param ctx the servlet context
     * @param req the HttpRequest object
     * @param resp the HttpResponse object
     * @exception ServletException if a servlet specific exception occurs
     * @exception IOException if fails to write respose back to the user
     * 
     */     
    public void doClearCart(
                           ServletContext ctx, 
                           HttpServletRequest req, 
                           HttpServletResponse resp,
                           Integer userID)
    throws ServletException, IOException {
        HttpSession session = req.getSession();
        ShoppingCart cart = getShoppingCart(session);
        cart.clear();
        postInvoke(req, cart);
        requestDispatch(ctx, req, resp, getPage(SHOPPINGCART_PAGE)); 
    }

    /**
     * Purchase all items from the current dealerships shopping cart of vehicles
     * Dispatch to the Spec order JSP for display 
     *
     * @param userID The User requesting to purchase their dealership's shopping cart
     * @param ctx the servlet context
     * @param req the HttpRequest object
     * @param resp the HttpResponse object
     * @exception ServletException if a servlet specific exception occurs
     * @exception IOException if fails to write respose back to the user
     * 
     */    
    public void doDeferedPurchase(
                          ServletContext ctx, 
                          HttpServletRequest req, 
                          HttpServletResponse resp,
                          Integer userID)
    throws ServletException, IOException {
        HttpSession session = req.getSession();
        ShoppingCart cart = getShoppingCart(session);
        if (cart.isEmpty()) {
            postInvoke(req, cart);
            requestDispatch(ctx, req, resp, getPage(SHOPPINGCART_PAGE));          
            return;
        }

        try {
            incrementVisit(session, KEY_ORDER_PLACED);
            Order order = sAction.doPurchase(userID, cart, 0, true, false);
            req.setAttribute(KEY_ORDER, order);
            req.setAttribute(KEY_RESULT, "has been added to your open orders.");
            cart.clear();
            requestDispatch(ctx, req, resp, getPage(ORDER_PAGE));
        } catch (InsufficientCreditException ex) {
            processError(ctx, req, resp, "doDeferedPurchase", 0, ex);
        } catch (Exception e) {
            processError(ctx, req, resp, "doDeferedPurchase", 1, e);
        }
    }

    /**
     * Purchase all items from the current dealerships shopping cart of vehicles
     * Dispatch to the Spec order JSP for display 
     *
     * @param userID The User requesting to purchase their dealership's shopping cart
     * @param ctx the servlet context
     * @param req the HttpRequest object
     * @param resp the HttpResponse object
     * @exception ServletException if a servlet specific exception occurs
     * @exception IOException if fails to write respose back to the user
     * 
     */    
    public void doPurchase(ServletContext ctx, 
                          HttpServletRequest req, 
                          HttpServletResponse resp,
                          Integer userID,
                          int location)
    throws ServletException, IOException {
        HttpSession session = req.getSession();
        ShoppingCart cart = getShoppingCart(session);
        if (cart.isEmpty()) {
            postInvoke(req, cart);
            requestDispatch(ctx, req, resp, getPage(SHOPPINGCART_PAGE));          
            return;
        }

        try {
            incrementVisit(session, KEY_ORDER_PLACED);
            session.setAttribute(KEY_PURCHASE_DEBITS, 
                    ((BigDecimal)session.getAttribute(KEY_PURCHASE_DEBITS))
                         .add(cart.getTotalPrice())); 
            Order order = sAction.doPurchase(userID, cart, location, false, false);
            req.setAttribute(KEY_ORDER, order);
            req.setAttribute(KEY_RESULT, "has been submitted for manufacturing.");
            cart.clear();
            requestDispatch(ctx, req, resp, getPage(ORDER_PAGE));
        } catch( InsufficientCreditException ex) {
            processError(ctx, req, resp, "doPurchase", 0, ex);
        } catch( Exception e) {
            processError(ctx, req, resp, "doPurchase", 1, e);
        }
    }        

    /**
     * Retrieve the current price quote for the given vehicle or return all
     * vehicles to the user
     * Dispatch to the Trade Quote JSP for display 
     *
     * @param vehicles The vehicle ID used to get the price quote
     * @param userID The ID of the user for who we are performing the query
     * @param ctx the servlet context
     * @param req the HttpRequest object
     * @param resp the HttpResponse object
     * @exception ServletException if a servlet specific exception occurs
     * @exception IOException if fails to write respose back to the user
     * 
     */
    public void doVehicleQuotes(
                               ServletContext ctx, 
                               HttpServletRequest req, 
                               HttpServletResponse resp, 
                               Integer userID, 
                               String vehicles,
                               int category) 
    throws ServletException, IOException {
        List<String> vehicleQuotes = null;
        if( vehicles != null ) {
            vehicleQuotes = new java.util.ArrayList<String>();
            java.util.StringTokenizer st = new java.util.StringTokenizer(vehicles, " ,");
            while( st.hasMoreElements() ) {
                String vehicle = st.nextToken();
                vehicleQuotes.add(vehicle); 
            }
        }
        try {
            HttpSession session = req.getSession();
            VehicleSearch itemInfo = sAction.getVehicleQuotes(vehicleQuotes, session, req.getParameter("browse"), category);
            incrementVisit(session, KEY_BROWSER_PAGE_VISIT);
            req.setAttribute("itemInfo", itemInfo);
            req.setAttribute("results", "");
            requestDispatch(ctx, req, resp, getPage(VEHICLEQUOTE_PAGE));
        } catch( Exception ex) {
            processError(ctx, req, resp, "doVehicleQuotes", 1, ex);
        }
    }


    /**
     * Sell a set of vehicles for the given dealership.
     * Dispatch to the Spec dealerinventory JSP for display 
     *
     * @param userID The User selling the vehicles
     * @param ctx the servlet context
     * @param req the HttpRequest object
     * @param resp the HttpResponse object
     * @exception ServletException if a servlet specific exception occurs
     * @exception IOException if fails to write respose back to the user
     * 
     */
    public void doSell(
                      ServletContext ctx, 
                      HttpServletRequest req, 
                      HttpServletResponse resp, 
                      Integer userID)
    throws ServletException, IOException {
        try {
            HttpSession session = req.getSession();
            Long holdingID = new Long(req.getParameter("vehicleToSell"));
            BigDecimal total = new BigDecimal((String)req.getParameter("total"));
            boolean sold = sAction.sell(userID, holdingID, false);
            if(sold){
                session.setAttribute("holdingsSold", new Integer(((Integer)session.getAttribute("holdingsSold")).intValue()+1)); 
                session.setAttribute("totalSalesProfits", ((BigDecimal)session.getAttribute("totalSalesProfits")).add(total));
                req.setAttribute("results", "");
            }else{
                req.setAttribute("results_sell", "*** NOTICE: The vehicles you tried to sell have already been sold! ***");
            }    
            this.doInventory(ctx, req, resp, userID); 
        } catch( Exception ex) {
            processError(ctx, req, resp, "doSell", 1, ex);
        }
    }

    public void doWelcome(
                         ServletContext ctx, 
                         HttpServletRequest req, 
                         HttpServletResponse resp, 
                         String status)
    throws ServletException, IOException {
        req.setAttribute("results", status);
        requestDispatch(ctx, req, resp, getPage(WELCOME_PAGE));
    }

    private void requestDispatch(
                                ServletContext ctx, 
                                HttpServletRequest req, 
                                HttpServletResponse resp, 
                                String page)
    throws ServletException, IOException {
        ctx.getRequestDispatcher(page).forward(req, resp);
    }

    /**
     * MFG Welcome Page.
     * Dispatch to the MFG Welcome Page
     * 
     * @param userID The User to login
     * @param ctx the servlet context
     * @param req the HttpRequest object
     * @param resp the HttpResponse object
     * @exception ServletException if a servlet specific exception occurs
     * @exception IOException if fails to write respose back to the user
     */
    public void doMfgWelcome(
                            ServletContext ctx, 
                            HttpServletRequest req, 
                            HttpServletResponse resp, 
                            String status)
    throws ServletException, IOException {
        req.setAttribute(KEY_RESULT, status);
        requestDispatch(ctx, req, resp, getPage(MFG_WELCOME_PAGE));
    }

   /**
    * Login a Spec Mfg User.
    * Dispatch to the SpecJMfgHome JSP for display
    * 
    * @param userID The User to login
    * @param ctx the servlet context
    * @param req the HttpRequest object
    * @param resp the HttpResponse object
    * @exception ServletException if a servlet specific exception occurs
    * @exception IOException if fails to write respose back to the user
    * 
    */
    public void doMfgLogin(
                          ServletContext ctx, 
                          HttpServletRequest req, 
                          HttpServletResponse resp, 
                          Integer userID) 
    throws ServletException, java.io.IOException {
        try {
            boolean loggedIn = sAction.login(userID);
            if( loggedIn ) {
                HttpSession session = req.getSession(true);
                if( session != null ) {
                    session.invalidate();
                    session = req.getSession(true); 
                }
                session.setAttribute(KEY_CUSTOMER_ID, userID);
                doMfgHome(ctx, req, resp, userID, "Ready to work");
            } else {
                req.setAttribute(KEY_RESULT, "Could not find account for " + userID);
                requestDispatch(ctx, req, resp, getPage(MFG_WELCOME_PAGE)); 
            }
        } catch( Exception ex) {
            processError(ctx, req, resp, "doMfgLogin", 1, ex);
        }
    }

    /**
     * Logout a Spec Mfg User
     * Dispatch to the Spec MFg Welcome JSP for display
     * 
     * @param userID The User to logout
     * @param ctx the servlet context
     * @param req the HttpRequest object
     * @param resp the HttpResponse object
     * @exception ServletException if a servlet specific exception occurs
     * @exception IOException if fails to write respose back to the user
     * 
     */
    public void doMfgLogout(
                           ServletContext ctx, 
                           HttpServletRequest req, 
                           HttpServletResponse resp, 
                           Integer userID)
    throws ServletException, IOException {
        HttpSession session = req.getSession();
        try {
            sAction.logout(session);
        } catch( Exception ex) {
            processError(ctx, req, resp, "doMfgLogout", 1, ex);
        }
        if( session != null ) {
            session.invalidate();
        }
        // Recreate Session object before writing output to the response
        // Once the response headers are written back to the client the opportunity
        // to create a new session in this request may be lost
        session = req.getSession(true);
        requestDispatch(ctx, req, resp, getPage(MFG_WELCOME_PAGE));
    }

    /**
     * Create the Mfg Home page
     * Dispatch to the MfgHome JSP for display
     *
     * @param ctx the servlet context
     * @param req the HttpRequest object
     * @param resp the HttpResponse object
     * @param results A short description of the results/success of this web request provided on the web page
     * @exception ServletException if a servlet specific exception occurs
     * @exception IOException if fails to write respose back to the user
     */
    public void doMfgHome(
                         ServletContext ctx, 
                         HttpServletRequest req, 
                         HttpServletResponse resp, 
                         Integer userID, 
                         String results)
    throws ServletException, java.io.IOException {
        try {
            req.setAttribute(KEY_RESULT, results);
            requestDispatch(ctx, req, resp, getPage(MFG_HOME_PAGE));
        } catch( Exception ex) {
            processError(ctx, req, resp, "doMfgHome", 1, ex);
        }
    }

    /**
     * MFG Schedule WorkOrder Page.
     * Dispatch to the MFG Schedule WorkOrder Page
     * 
     * @param userID The User to login
     * @param ctx the servlet context
     * @param req the HttpRequest object
     * @param resp the HttpResponse object
     * @exception ServletException if a servlet specific exception occurs
     * @exception IOException if fails to write respose back to the user
     * 
     */
    public void doMfgScheduleWO(
                               ServletContext ctx, 
                               HttpServletRequest req, 
                               HttpServletResponse resp)
    throws ServletException, IOException {
        List<String> assemblyIds;
        int location=0,qty=0,oLineId=0,salesId=0;
        Calendar dueDate=null;
        boolean errorFound = false;

        req.setAttribute("nextAction","mfgschedulewo");

        String str_assemblyId = req.getParameter("assemblyId");
        String str_qty = req.getParameter("qty");
        String str_location = req.getParameter("location");
        String str_oLineId = req.getParameter("oLineId");
        String str_salesId = req.getParameter("salesId");
        String str_dueDate = req.getParameter("dueDate");

        if  (str_location != null ) {
          location = Integer.parseInt(str_location);
        }
        if( str_assemblyId == null || str_qty == null ) {
            try {
                assemblyIds = sAction.getAssemblyIds();
                req.setAttribute("results","");
                req.setAttribute("assemblyIds", assemblyIds);
                requestDispatch(ctx, req, resp, getPage(MFG_SCHEDULEWO_PAGE));
                return;
            } catch( Exception ex) {
                processError(ctx, req, resp, "doMfgScheduleWO", 1, ex);
            }
        } else {
            try {
                qty = Integer.parseInt(str_qty);
                SimpleDateFormat s = new SimpleDateFormat("yyyy-MM-dd");
                if( str_dueDate != null ) {
                    dueDate = Calendar.getInstance();
                    dueDate.setTime(s.parse(str_dueDate));
                }
                if( str_salesId != null ) salesId = Integer.parseInt(str_salesId);
                if( str_oLineId != null ) oLineId = Integer.parseInt(str_oLineId);
            } catch( NumberFormatException e ) {
                req.setAttribute(KEY_RESULT,"Invalid Quantity: "+str_qty+".");
                errorFound = true;
            } catch( Exception e ) {
                req.setAttribute("results","Invalid Parameter.");
                errorFound = true;
            }
            if (errorFound) {
                try {
                    assemblyIds = sAction.getAssemblyIds();
                    req.setAttribute("assemblyIds", assemblyIds);
                    requestDispatch(ctx, req, resp, getPage(MFG_SCHEDULEWO_PAGE));
                    return;
                } catch( Exception ex) {
                    processError(ctx, req, resp, "doMfgScheduleWO", 1, ex);
                }

            }

            try {
                WorkOrder wo = null;
                if( str_oLineId == null || str_salesId == null || str_dueDate == null ) {
                    wo = sAction.scheduleWorkOrder(str_assemblyId, qty, location, dueDate);
                } else {
                    wo = sAction.scheduleWorkOrder(salesId,oLineId,str_assemblyId,qty, location, dueDate);
                }
                req.setAttribute("results","Scheduled order Work Order: "+wo.getId()+".");
                req.setAttribute("defaultWoID", Integer.toString(wo.getId()));
                req.setAttribute("displayAction","Update");
                req.setAttribute("nextAction","mfgupdatewo");
            } catch( Exception ex) {
                processError(ctx, req, resp, "doMfgScheduleWO", 1, ex);
            }
        }
        requestDispatch(ctx, req, resp, getPage(MFG_WORKORDER_PAGE));
        return;
    }


    /**
     * MFG Cancel WorkOrder Page.
     * Dispatch to the MFG Cancel WorkOrder Page
     * 
     * @param userID The User to login
     * @param ctx the servlet context
     * @param req the HttpRequest object
     * @param resp the HttpResponse object
     * @exception ServletException if a servlet specific exception occurs
     * @exception IOException if fails to write respose back to the user
     * 
     */
    public void doMfgCancelWO(
                             ServletContext ctx, 
                             HttpServletRequest req, 
                             HttpServletResponse resp)
    throws ServletException, IOException {
        Integer woID = null;
        Integer location = null;

        req.setAttribute("displayAction","Cancel");
        req.setAttribute("nextAction","mfgcancelwo");

        String str_woID = req.getParameter("order_id");
        if( str_woID == null ) {
            req.setAttribute("results","");
            requestDispatch(ctx, req, resp, getPage(MFG_WORKORDER_PAGE));
            return;
        }
        String str_location = req.getParameter("location");
        if  (str_location == null ) {
          str_location = "0";
        }  
        try {
            woID = Integer.valueOf(str_woID);
            location = Integer.valueOf(str_location);
        } catch( NumberFormatException e ) {
            req.setAttribute("results","Invalid Work Order: "+str_woID+" at location: "+location+".");
            requestDispatch(ctx, req, resp, getPage(MFG_WORKORDER_PAGE));
            return;
        }

        try {

            if( sAction.cancelWorkOrder(woID, location) ) {
                req.setAttribute("results","Cancelled order Work Order: "+str_woID+" at location: "+location+".");

            } else {
                req.setAttribute("results","Work Order Number: "+str_woID+" at location: "+location+" can't be Cancelled.");

            }
        } catch( Exception ex) {
            processError(ctx, req, resp, "doMfgCancelWO", 1, ex);
        }

        requestDispatch(ctx, req, resp, getPage(MFG_WORKORDER_PAGE));
        return;
    }


    /**
     * MFG Complete WorkOrder Page.
     * Dispatch to the MFG Complete WorkOrder Page
     * 
     * @param userID The User to login
     * @param ctx the servlet context
     * @param req the HttpRequest object
     * @param resp the HttpResponse object
     * @exception ServletException if a servlet specific exception occurs
     * @exception IOException if fails to write respose back to the user
     * 
     */
    public void doMfgCompleteWO(
                               ServletContext ctx, 
                               HttpServletRequest req, 
                               HttpServletResponse resp)
    throws ServletException, IOException {
        Integer woID = null;
        Integer location = null;

        req.setAttribute("displayAction","Complete");
        req.setAttribute("nextAction","mfgcompletewo");

        String str_woID = req.getParameter("order_id");
        if( str_woID == null ) {
            req.setAttribute("results","");
            requestDispatch(ctx, req, resp, getPage(MFG_WORKORDER_PAGE));
            return;
        }
        String str_location = req.getParameter("location");
        if  (str_location == null ) {
          str_location = "0";
        }
        try {
            woID = Integer.valueOf(str_woID);
            location = Integer.valueOf(str_location);
        } catch( NumberFormatException e ) {
            req.setAttribute("results","Invalid Work Order: "+str_woID+".");
            requestDispatch(ctx, req, resp, getPage(MFG_WORKORDER_PAGE));
            return;
        }

        try {

            if( sAction.completeWorkOrder(woID, location) ) {
                req.setAttribute("results","Completed order Work Order: "+str_woID+".");

            } else {
                req.setAttribute("results","Work Order Number: "+str_woID+" can't be Completed.");

            }
        } catch( Exception ex) {
            processError(ctx, req, resp, "doMfgCompleteWO", 1, ex);
        }

        requestDispatch(ctx, req, resp, getPage(MFG_WORKORDER_PAGE));
        return;
    }


    /**
     * MFG Update WorkOrder Page.
     * Dispatch to the MFG Update WorkOrder Page
     * 
     * @param userID The User to login
     * @param ctx the servlet context
     * @param req the HttpRequest object
     * @param resp the HttpResponse object
     * @exception ServletException if a servlet specific exception occurs
     * @exception IOException if fails to write respose back to the user
     * 
     */
    public void doMfgUpdateWO(
                             ServletContext ctx, 
                             HttpServletRequest req, 
                             HttpServletResponse resp)
    throws ServletException, IOException {
        Integer woID = null;
        Integer location = null;

        req.setAttribute("displayAction","Update");
        req.setAttribute("nextAction","mfgupdatewo");

        String str_woID = req.getParameter("order_id");
        if( str_woID == null ) {
            req.setAttribute("results","");
            requestDispatch(ctx, req, resp, getPage(MFG_WORKORDER_PAGE));
            return;
        }
        String str_location = req.getParameter("location");
        if  (str_location == null ) {
            str_location = "0";
        }
        try {
            woID = Integer.valueOf(str_woID);
            location = Integer.valueOf(str_location);
        } catch( NumberFormatException e ) {
            req.setAttribute("results","Invalid Work Order: "+str_woID+".");
            requestDispatch(ctx, req, resp, getPage(MFG_WORKORDER_PAGE));
            return;
        }

        try {
            if( sAction.updateWorkOrder(woID, location) ) {
                req.setAttribute("results","Updated order Work Order: " + 
                                str_woID+" at location: " + str_location + 
                                ". The current status of the order is " + 
                                sAction.getWorkOrderStatus(woID, location)+".");
                req.setAttribute("defaultWoID",str_woID);

            } else {
                req.setAttribute("results","Work Order Number: " + str_woID + 
                                " at location: " + str_location + 
                                " can't be Updated. It is in the final stage " + 
                                sAction.getWorkOrderStatus(woID, location)+".");
                req.setAttribute("defaultWoID",str_woID);
                req.setAttribute("displayAction","Complete");
                req.setAttribute("nextAction","mfgcompletewo");
            }
        } catch( Exception ex) {
            processError(ctx, req, resp, "doMfgUpdateWO", 1, ex);
        }

        requestDispatch(ctx, req, resp, getPage(MFG_WORKORDER_PAGE));
        return;
    }
 
    /**
     * Check for the next queued work order and start executing it.
     * 
     * @param ctx
     * @param req
     * @param resp
     * @throws ServletException 
     * @throws IOException 
     */
    public void doMfgCheckWork(ServletContext ctx, HttpServletRequest req,
        HttpServletResponse resp) throws IOException, ServletException {
      
      try {
        req.setAttribute("nextAction","mfgcheckwork");

        String str_location = req.getParameter("location");

        // if this is our first pass through
        // the page just redirect..
        if (str_location==null) {
          req.setAttribute("results","");
          requestDispatch(ctx, req, resp, getPage(MFG_WORKORDER_CHECK_PAGE));
          return;
        }
        
        req.setAttribute("results","There are no pending work requests");
        req.setAttribute("displayAction","Schedule");
        requestDispatch(ctx, req, resp, getPage(MFG_WORKORDER_PAGE));          
      } catch( Exception ex) {
        processError(ctx, req, resp, "doMfgScheduleWO", 1, ex);
      }
    
    } 
    
	/**
	 * Tests that the correct transactional attributes are applied to ensure data integrity
	 * 
	 * @param userID The User to login
	 * @param ctx the servlet context
	 * @param req the HttpRequest object
	 * @param resp the HttpResponse object
	 * @exception ServletException if a servlet specific exception occurs
	 * @exception IOException if fails to write respose back to the user
	 * 
	 */
	public void doAtomicityTests(
							ServletContext ctx, 
							HttpServletRequest req, 
							HttpServletResponse resp)
							throws ServletException, IOException {
								
		boolean test1, test2, test3;
		try{
			test1 = sAction.atomicityTestOne();
			req.setAttribute("test1", new Boolean(test1));	
		}catch(Exception e){
			req.setAttribute("test1", new Boolean(false));
		}
		try{
			test2 = sAction.atomicityTestTwo();	
			req.setAttribute("test2", new Boolean(test2));	
		}catch(Exception e){
			req.setAttribute("test2", new Boolean(false));
			
		}
		try{
			test3 = sAction.atomicityTestThree();	
			req.setAttribute("test3", new Boolean(test3));	
		}catch(Exception e){
			req.setAttribute("test3", new Boolean(false));
		}
		requestDispatch(ctx, req, resp, getPage(ATOMICITY_PAGE)); 
		return;    
	}
	
	public void doCacheTest(
							ServletContext ctx, 
							HttpServletRequest req, 
							HttpServletResponse resp,
							int cacheTestId,
							String itemToUpdate)
							throws ServletException, IOException {
		Item idb = null;						
		if(cacheTestId==0){
			BigDecimal newValue = null;
			// This is where we want to hydrate the cache and then update the value in the
			// database so later on we can check that it has been updated in the JPA layer
			try{
                            System.out.println("Cache Timeout test item id is " + itemToUpdate);

				idb = sAction.getItemJPA(itemToUpdate);
                                System.out.println("Cache Timeout price before update is " + idb.getPrice());
				newValue = idb.getPrice().add(new BigDecimal(2000.00));
                                System.out.println("Cache Timeout price after update should be " + newValue);
				sAction.updateItemJDBC(itemToUpdate, newValue);
			}catch(Exception e){
				req.setAttribute("cacheTest", null);
			}
			req.setAttribute("cacheTest", newValue);
		}else{
                        // In the second part of the cache test we use JPA to retrieve the same item we previously updated
                        // and check to ensure that the app server has reloaded the price from the database
                        BigDecimal newprice = new BigDecimal(0.0);
			try {		 
                                 idb= sAction.getItemJPA(itemToUpdate);
                                 newprice = idb.getPrice();
				 req.setAttribute("cacheTest", newprice );
			} catch (Exception e) {
                            e.printStackTrace(); 
				req.setAttribute("cacheTest", null);
			}
		}
		requestDispatch(ctx, req, resp, getPage(CACHETEST_PAGE)); 
		return;
	}
	
	/**
	 * Gets the ShoppingCart given the customerId.
	 * If the ShoppingCart already exists in the session then simply returns
	 * it. Otherwise, if a Customer is associated with the session, then
	 * creates a ShoppingCart for that Customer.
	 *  
	 * @param session
	 * @param customerId
	 * @return
	 */
	ShoppingCart getShoppingCart(HttpSession session) {
        ShoppingCart cart = (ShoppingCart) session.getAttribute(KEY_CART);
        if( cart == null ) {
            cart = new ShoppingCart(); 
            session.setAttribute(KEY_CART, cart);
        }
        return cart;
	}

  public void doSupplierSetAllURLs(ServletContext ctx, HttpServletRequest req,
      HttpServletResponse resp) throws ServletException, IOException {
    String supplierWebServiceURL = 
      req.getParameter("supp_ws_url");
    String supplierReplyURL = 
      req.getParameter("supp_reply_url");
    sAction.setAllSupplierURLs(supplierWebServiceURL, supplierReplyURL);
    Collection<Supplier> suppliers = sAction.getAllSuppliers();
    req.setAttribute(KEY_SUPPLIERS, suppliers);
    requestDispatch(ctx, req, resp, getPage(SUPPLIER_HOME_PAGE));
  }

  public void doSupplierWelcome(ServletContext ctx, HttpServletRequest req,
      HttpServletResponse resp) throws ServletException, IOException {
    Collection<Supplier> suppliers = sAction.getAllSuppliers();
    req.setAttribute(KEY_SUPPLIERS, suppliers);
    requestDispatch(ctx, req, resp, getPage(SUPPLIER_HOME_PAGE));
  }

 
}
