/*
 *   SPECjEnterprise2010 - a benchmark for enterprise middleware
 *  Copyright 1995-2010 Standard Performance Evaluation Corporation
 *   All Rights Reserved
 *
 * This source code is provided as is, without any express or implied warranty.
 *
 *  History:
 *  Date        ID, Company               Description
 *  ----------  -----------------------   ---------------------------------------------------------------
 *  2007/10/02  Bernhard Riedhofer, SAP   Created, integration of loader into SPECjAppServer2007 application
 *  2008/07/30  Anoop Gupta, Oracle       Added numLocation for parts
 */
package org.spec.jent.loader;

import java.lang.reflect.Constructor;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.SortedSet;
import java.util.logging.Logger;

import javax.jms.JMSException;
import javax.naming.NamingException;

import org.spec.jent.common.util.RandNum;
import org.spec.jent.common.util.Utils;
import org.spec.jent.loader.entity.Loader;
import org.spec.jent.loader.entity.Section;

/*
 * Super class of the domain loader classes.
 */
public abstract class Load  {

    private static final String CLASSNAME = Load.class.getName();
    private static final Logger logger = Logger.getLogger(CLASSNAME);
    
    static final int LOAD_IN_ONE_SECTION = 1;

    final LoadTracker loadTracker;
    final Loader loader;

    final String dbKey;
    final String domain;
    int sectionId;
    
    final int scale;
    final int numCustomers;
    final int numAssemblies;
    static int numLocations;
    final int numItems;
    static int itemsPerTxRate;
    static int maxItemsPerLoc;
    final int numOrders;
    
    final SeedGenerator seedGen;
    final RandNum r;
    
    final InfrequentChecker infrequentChecker = new InfrequentChecker();
    boolean isStopping = false;
    
    @SuppressWarnings("unchecked")
    public static void loadNextSection(LoadTracker loadTracker) throws Throwable {
        Section section = null; 
        try {
           section = loadTracker.getNextSection();
        } catch (Throwable t) {
          // dangerous path since we cannot do something without having access to loadTracker
          // -> has to be handeled by audit which checks if database is loaded correctly 
          if (loadTracker == null) {
            t.printStackTrace();
          } else {
            loadTracker.addMessage("Processing LoaderMDB failed to getNextSection.", t);
          }
          throw t;
        }
        try {
            String className = Load.class.getName() + section.getDomain();
            Class<Load> clazz = (Class<Load>)Class.forName(className);
            Constructor<Load> c = clazz.getConstructor(LoadTracker.class);
            Load load = c.newInstance(loadTracker);
            load.loadTableSection(section);
            if (load.isStopping()) {
                loadTracker.changeSectionState(section, Section.State.CANCELED);
            }
        } catch (Throwable t) {
            loadTracker.changeSectionState(section, Section.State.FAILED, t);
            throw t;
        }
    }

    Load(LoadTracker loadTracker, String dbKey, String domain) {
        this.loadTracker = loadTracker;
        this.loader = loadTracker.getLoader();
        this.dbKey = dbKey;
        this.domain = domain;
        sectionId = Section.FIRST_SECTION_NUMBER;

        scale = loader.getDbIR();
        numCustomers = Helper.getNumCustomers(scale);
        numAssemblies = loader.getItemCount();
        numLocations = Helper.getNumLocations(scale * loader.getItemsPerTxRate(), numAssemblies);
        numOrders = Helper.getNumOrders(scale);
        numItems = loader.getItemCount();
        itemsPerTxRate = Utils.itemsPerTxRate;
        maxItemsPerLoc = Utils.maxItemsPerLoc;
        
        // see class Component for an explanation why this has to be checked
        if (numAssemblies >= Component.SHARED_ASSEMBLY_BLOCK_SIZE
                && numAssemblies % Component.SHARED_ASSEMBLY_BLOCK_SIZE != 0) {
            throw new RuntimeException("The following condition is not met:\n"
                    + "numAssemblies % Component.SHARED_ASSEMBLY_BLOCK_SIZE == 0\n" + "numAssemblies=" + numAssemblies
                    + ", SHARED_ASSEMBLY_BLOCK_SIZE=" + Component.SHARED_ASSEMBLY_BLOCK_SIZE);
        }

        seedGen = new SeedGenerator(loader.getRandomRootSeed());
        r = new RandNum();
    }

    Connection getConnection() throws SQLException, NamingException {
        return getConnection(dbKey);
    }
    
    Connection getConnection(String dbKey) throws SQLException, NamingException {
        if (loader.isFlatFilesGenerated()) {
            return FlatFileConnection.newConnectionProxy(sectionId, loader.getFlatFilesDirectory(), loader
                    .getFlatFilesDelimiter());
        }
        if (loader.isStandAlone()) {
            return createNewConnection();
        }
        return DatabaseHelper.getConnection(dbKey);
    }
    
    Connection createNewConnection() {
        try {
            Class.forName(loader.getDriverClass());
        } catch (ClassNotFoundException e) {
            throw new IllegalArgumentException("Unable to load driver " + loader.getDriverClass(), e);
        }
        try {
            logger.fine("Connecting to " + loader.getDBURL() + " as " + loader.getDBUser() + ":"
                    + loader.getDBPassword());
            Connection retval = DriverManager.getConnection(loader.getDBURL(), loader.getDBUser(), loader
                    .getDBPassword());
            retval.setAutoCommit(false);
            return retval;
        } catch (SQLException e) {
            throw new IllegalArgumentException("Unable to connect to database " + loader.getDBURL(), e);
        }
    }
    
    void cleanTables(Section section) throws SQLException, NamingException {
        for (String table : section.getTableNamesArray()) {
            if (isStopping()) return;
            DatabaseHelper.clean(getConnection(), table);
        }
    }

    void loadSequence(String id, long nextSeq) throws SQLException, NamingException, JMSException {
        if (isStopping()) return;
        DatabaseHelper.loadSequence(getConnection(), id, nextSeq);
    }

    void loadSequence(String id, int nextSeq) throws SQLException, NamingException, JMSException {
        loadSequence(id, (long)nextSeq);
    }

    void genSections(SortedSet<Section> sections, String tables, int numSteps) throws CloneNotSupportedException {
      genSections(sections, tables, numSteps, 1);
    }
    
    void genSections(SortedSet<Section> sections, String tables, int numSteps, int multipleStepsPerSection)
            throws CloneNotSupportedException {
        Section section = new Section();
        section.setDomain(domain);
        section.setTableNames(tables);
        section.setState(Section.State.INITIAL);
        // to get maximum parallelism even if there is only one table loaded:
        // for each table there are as many sections as threads which should be started due the parallelism parameter
        int numSections;
        if (numSteps < multipleStepsPerSection) {
            numSections = 1;
        } else {
            numSections = Helper.calcRoundedNumSections(numSteps / multipleStepsPerSection, loader.getParallelism());
        }
        int stepsPerSection = numSteps / numSections;
        section.setNumSteps(stepsPerSection);
        for (int sectionId = 0; sectionId < numSections; sectionId++) {
            section.setSectionId(sectionId + Section.FIRST_SECTION_NUMBER);
            section.setStartStep((sectionId * stepsPerSection) + 1);
            sections.add(section.clone());
        }
    }

    abstract void genSections(SortedSet<Section> sections) throws CloneNotSupportedException;
    abstract void loadSequences() throws SQLException, NamingException, JMSException;

    /*
     * Loading a section of a table.
     * @return true if finished successfully otherwise loading is stopping
     * @throws exception is thrown if loading failed
     */
    abstract void loadTableSection(final String tables, final int start, final int num)  throws SQLException, NamingException, JMSException;
    
    public void loadTableSection(Section section) throws SQLException, NamingException, JMSException {
        sectionId = section.getSectionId();
        // make sure that there are not more MDBs working than parallelism parameter indicates 
        loadTracker.waitMaxParallelism(section);

        // deleting table(s) (during loading first section only,  othwerwise wait till tables are deleted)
        if (isStopping()) return;
        if (!loader.isFlatFilesGenerated()) {
            if (section.isFirstSection()) {
                loadTracker.changeSectionState(section, Section.State.DELETING);
                cleanTables(section);
            } else {
                loadTracker.waitForTablesDeleted(section);
            }
        }

        if (isStopping()) return;
        loadTracker.changeSectionState(section, Section.State.LOADING);
        loadTableSection(section.getTableNames(), section.getStartStep(), section.getNumSteps());
        if (isStopping()) return;
        loadTracker.changeSectionState(section, Section.State.FINISHED_SUCCESSFULLY);
    }

    public boolean isStopping() {
        if (isStopping) {
            return true;
        }
        isStopping = loadTracker.getLoader().getState().isStopping();
        return isStopping;
    }

    public class InfrequentChecker {
        private static final int MAX_SKIP_COUNTER = 10;
        long skipCounter = MAX_SKIP_COUNTER; // force check for first invocation of isStopping()
        long lastTimeChecked = 0;

        public boolean isStopping() throws NamingException, JMSException {
            if (skipCounter < MAX_SKIP_COUNTER) {
                skipCounter++;
                return false;
            }
            skipCounter = 0;
            long currentTime = System.currentTimeMillis();
            if (currentTime - lastTimeChecked < WebLoadTracker.SLEEP_TIME_IN_MILLIS) {
                return false;
            }
            lastTimeChecked = currentTime;
            return Load.this.isStopping();
        }
    }
}
