/*
 *   SPECjEnterprise2010 - a benchmark for enterprise middleware
 *  Copyright 1995-2010 Standard Performance Evaluation Corporation
 *   All Rights Reserved
 *
 * This source code is provided as is, without any express or implied warranty.
 *
 *  History:
 *  Date        ID, Company               Description
 *  ----------  ------------------------  ------------------------------------------------------
 *  2002/03/24  ramesh, SUN Microsystem   Created
 *  2001/09/17  Matt Hogstrom, IBM        Modified the customer selection to avoid
 *                                        creating orders for customers with bad credit.
 *  2001/11/06  Matt Hogstrom, IBM        Modified  SQLDate to use a timestmp instead
 *                                        of a java.sql.Date to match the actual table
 *                                        column type.
 *  2002/04/12  Matt Hogstrom, IBM        Conversion from ECperf 1.1 to SPECjAppServer2001*
 *  2002/07/10  Russel Raymundo, BEA      Conversion from SPECjAppServer2001 to
 *                                        SPECjAppServer2002 (EJB2.0).
 *  2003/04/03  Russel Raymundo, BEA      Explicitly specified table columns names for
 *                                        inserts. Enabled Distributed Database loading.
 *  2003/03/21  BSthanikam, Oracle        Do not load O_Customer table. Change loading of
 *                                        o_orderline to load ol_total_value column too.
 *  2003/05/29  Samuel Kounev, Darmstadt  Changed to make sure all large orderlines are
 *                                        created with status 3 (completed).
 *  2003/11/06  Samuel Kounev, Darmstadt  Modified database scaling rules (see osgjava-5681).
 *  2003/11/24  Samuel Kounev, Darmstadt  Modified to populate new i_category column of the
 *                                        o_item table (see osgjava-6001).
 *  2003/11/25  Samuel Kounev, Darmstadt  Modified database scaling rules as per osgjava-5891.
 *  2004/01/13  Rafay Khawaja, Borland    Re-arranged code, and added some threads for cocurrent
 *                                        loading where possible.
 *  2004/02/20  Rafay Khawaja, Borland    Addressed issue with assigning correct categories to
 *                                        ranges of ItemIds when loading using multiple threads.
 *  2007/10/02  Bernhard Riedhofer, SAP   Created, integration of loader into SPECjAppServer2007 application
 *  2008/07/23  Anoop Gupta, Oracle       Have fixed set of assemblies in O_ITEMS and Assembly 
                                          shares components
 */
package org.spec.jent.loader;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;

import javax.jms.JMSException;
import javax.naming.NamingException;

import org.spec.jent.common.Constants;
import org.spec.jent.loader.entity.Section;

/*
 * Loads the tables of orders domain.
 */
class LoadOrders extends Load implements Constants {
    
    /* name of the domain must be the same as in the name of this class since this domain name is used for reflection */
    static final String ORD_DOMAIN = "Orders";

    private static final int ORDER_SIZE = 5;
    private static final int ITEMS_PER_CATEGORY = 200;

    private static final String CUSTOMER_TABLE_NAME = "O_CUSTOMER";
    private static final String CUSTOMERINVENTORY_TABLE_NAME = "O_CUSTINVENTORY";
    private static final String ITEM_TABLE_NAME = "O_ITEM";
    private static final String ORDERS_TABLE_NAMES = "O_ORDERS,O_ORDERLINE";

    private static Map<Integer, Double> assemblyPrices;
    private static int threadsUsingAssemblyPrices = 0;
    private static Set<Integer> badCreditCustomers;
    private static int threadsUsingBadCreditCustomers = 0;
    
    private static synchronized void useAssemblyPrices(int numAssemblies, int numAssembliesPerThread, SeedGenerator seedGen) {
        if (threadsUsingAssemblyPrices == 0) {
            assemblyPrices = Assembly.getAssemblyPrices(numAssemblies, numAssembliesPerThread, seedGen);
        }
        threadsUsingAssemblyPrices++;
    }

    // Free it for GC if there are no more users.
    // We want to use only one copy per VM.
    // This technique is needed since the threads could be executed on more than one VM.
    private static synchronized void releaseAssemblyPrices() {
        threadsUsingAssemblyPrices--;
        if (threadsUsingAssemblyPrices == 0) {
            assemblyPrices = null;
        }
    }

    private static synchronized void useBadCreditCustomers(int numCustomers, SeedGenerator seedGen) {
        if (threadsUsingBadCreditCustomers == 0) {
            badCreditCustomers = Customer.getBadCreditCustomers(1, numCustomers,
                    EVERY_XTH_CUSTOMER_WITH_BAD_CREDIT, seedGen);
        }
        threadsUsingBadCreditCustomers++;
    }

    // Free it for GC if there are no more users.
    // We want to use only one copy per VM.
    // This technique is needed since the threads could be executed on more than one VM.
    private static synchronized void releaseBadCreditCustomers() {
        threadsUsingBadCreditCustomers--;
        if (threadsUsingBadCreditCustomers == 0) {
            badCreditCustomers = null;
        }
    }
    
    public LoadOrders(LoadTracker loadTracker) {
        super(loadTracker, DatabaseHelper.ORDER_KEY, ORD_DOMAIN);
    }

    @Override
    void genSections(SortedSet<Section> section) throws CloneNotSupportedException {
        genSections(section, CUSTOMER_TABLE_NAME, numCustomers);
        genSections(section, CUSTOMERINVENTORY_TABLE_NAME, numCustomers);
        genSections(section, ITEM_TABLE_NAME, numAssemblies, Component.SHARED_ASSEMBLY_BLOCK_SIZE);
        genSections(section, ORDERS_TABLE_NAMES, numOrders);
    }

    @Override
    void loadSequences() throws SQLException, NamingException, JMSException {
        loadSequence("customer", numCustomers + 1);
        loadSequence("inventory", (long)MAX_INVENTORY_PER_CUSTOMER * (long)numCustomers + 1);
        loadSequence("order", numOrders + 1);
    }

    @Override
    void loadTableSection(final String tables, final int start, final int num) throws SQLException, NamingException, JMSException {
        if (tables.equals(CUSTOMER_TABLE_NAME)) {
            loadCustomer(start, num);
        } else if (tables.equals(CUSTOMERINVENTORY_TABLE_NAME)) {
            loadCustomerInventory(start, num);
        } else if (tables.equals(ITEM_TABLE_NAME)) {
            loadItem(start, num);
        } else if (tables.equals(ORDERS_TABLE_NAMES)) {
            loadOrders(start, num);
        } else {
            throw new IllegalArgumentException("Table(s) " + tables + " not found.");
        }
    }

    private void loadCustomer(final int start, final int num) throws SQLException, NamingException, JMSException {
        final int end = start + num;
        final Connection dbConnection = getConnection();
        try {
            // The column ordering here and in the schema (schema_O.sql) must match, so that flat file generation properly maps to columns
            final PreparedStatement cs = dbConnection
                    .prepareStatement("insert into O_CUSTOMER (C_ID, C_FIRST, C_LAST, "
                            + "C_STREET1, C_STREET2, C_CITY, C_STATE, " + "C_COUNTRY, C_ZIP, C_PHONE, "
                            + "C_CONTACT, C_SINCE, C_BALANCE, C_CREDIT, C_CREDIT_LIMIT, C_YTD_PAYMENT, C_VERSION) "
                            + "values (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)");
            try {
                Customer.RandomGenerator customerGen = new Customer.RandomGenerator();
                Set<Integer> badCreditCustomers = Customer.getBadCreditCustomers(start, num,
                        EVERY_XTH_CUSTOMER_WITH_BAD_CREDIT, seedGen);
                int batchCount = 0;
                try {
                    for (int cid = start; cid < end; cid++) {
                        if (infrequentChecker.isStopping()) return;
                        Customer cust = customerGen.createCustomer();
                        if (badCreditCustomers.contains(cid)) {
                            cust.setBadCredit();
                        }
                        Address address = cust.getAddress();
                        cs.setInt(1, cid);
                        cs.setString(2, cust.getFirst());
                        cs.setString(3, cust.getLast());
                        cs.setString(4, address.getStreet1());
                        cs.setString(5, address.getStreet2());
                        cs.setString(6, address.getCity());
                        cs.setString(7, address.getState());
                        cs.setString(8, address.getCountry());
                        cs.setString(9, address.getZip());
                        cs.setString(10, address.getPhone());
                        cs.setString(11, cust.getContact());
                        cs.setDate(12, cust.getSince());
                        cs.setDouble(13, cust.getBalance());
                        cs.setString(14, cust.getCredit());
                        cs.setDouble(15, cust.getCreditLimit());
                        cs.setDouble(16, cust.getYtdPayment());
                        cs.setInt(17, 0);
                        batchCount = DatabaseHelper.addAndExecuteBatchIfFull(dbConnection, batchCount, cs);
                    }
                } finally {
                    DatabaseHelper.executeBatch(dbConnection, batchCount, cs); // load remaining rows
                }
            } finally {
                cs.close();
            }
        } finally {
            dbConnection.close();
        }
    }

    private void loadCustomerInventory(final int start, final int num) throws SQLException, NamingException,
            JMSException {
        final int end = start + num;
        final Connection dbConnection = getConnection();
        try {
            // The column ordering here and in the schema (schema_O.sql) must match, so that flat file generation properly maps to columns
            final PreparedStatement cs = dbConnection.prepareStatement("insert into O_CUSTINVENTORY"
                    + " (CI_ID, CI_CUSTOMERID, CI_ITEMID, CI_QUANTITY, CI_VALUE, CI_VERSION)" 
                    + " values (?,?,?,?,?,?)");
            try {
                long customerInventorId = (long)MAX_INVENTORY_PER_CUSTOMER * ((long)start - 1) + 1;
                int batchCount = 0;
                try {
                    useAssemblyPrices(numAssemblies, Helper.getNumPerThread(numAssemblies, loader.getParallelism()),
                            seedGen);
                    try {
                        for (int customerId = start; customerId < end; customerId++) {
                            if (infrequentChecker.isStopping()) return;
                            final int numCars = r.random(MIN_INVENTORY_PER_CUSTOMER, MAX_INVENTORY_PER_CUSTOMER);
                            for (int k = 0; k < numCars; k++) {
                                final int assemblyNum = r.random(1, numAssemblies);
                                final String pId = Assembly.getAssemblyId(assemblyNum);
                                final int pQty = r.random(MIN_INVENTORY_VEHICLE_COUNT, MAX_INVENTORY_VEHICLE_COUNT);
                                final double value = assemblyPrices.get(assemblyNum).doubleValue() * pQty
                                        * r.drandom(0.3, 1);
                                cs.setLong(1, customerInventorId++);
                                cs.setInt(2, customerId);
                                cs.setString(3, pId);
                                cs.setInt(4, pQty);
                                cs.setDouble(5, value);
                                cs.setInt(6, 0);
                                batchCount = DatabaseHelper.addAndExecuteBatchIfFull(dbConnection, batchCount, cs);
                            }
                        }
                    } finally {
                        releaseAssemblyPrices();
                    }
                } finally {
                    DatabaseHelper.executeBatch(dbConnection, batchCount, cs); // load remaining rows
                }
            } finally {
                cs.close();
            }
        } finally {
            dbConnection.close();
        }
    }
    
    private void loadItem(final int start, final int num) throws SQLException, NamingException, JMSException {
        final int end = start + num;
        final Connection dbConnection = getConnection();
        try
        {
            // The column ordering here and in the schema (schema_O.sql) must match, so that flat file generation properly maps to columns
            final PreparedStatement cs = dbConnection.prepareStatement("insert into O_ITEM"
                    + " (I_ID, I_NAME, I_DESC, I_PRICE, I_DISCOUNT, I_CATEGORY, I_VERSION)" + " values (?,?,?,?,?,?,?)");
            try {
                int assemblyBlock = start/num;
                final Assembly.RandomGenerator assemblyGen = new Assembly.RandomGenerator(seedGen
                        .getAssemblySeed(assemblyBlock), start);
                int batchCount = 0;
                try {
                    for (int j = start; j < end; j++) {
                        if (infrequentChecker.isStopping()) return;
                        Assembly assembly = assemblyGen.createAssembly();
                        final String pId = assembly.getId();
                        final float iDiscount = (float) (r.drandom(0.00, 0.70));
                        final String idString = pId.substring(10);
                        final int idFromStr = Integer.parseInt(idString);
                        final int iCategory = (idFromStr - 1) / ITEMS_PER_CATEGORY;
                        cs.setString(1, pId);
                        cs.setString(2, assembly.getName());
                        cs.setString(3, assembly.getDesc());
                        cs.setDouble(4, assembly.getPrice());
                        cs.setFloat(5, iDiscount);
                        cs.setInt(6, iCategory);
                        cs.setInt(7, 0);
                        batchCount = DatabaseHelper.addAndExecuteBatchIfFull(dbConnection, batchCount, cs);
                    }
                } finally {
                    DatabaseHelper.executeBatch(dbConnection, batchCount, cs); // load remaining rows
                }
            } finally {
                cs.close();
            }
        } finally {
            dbConnection.close();
        }
    }

    private void loadOrders(final int start, final int num)
            throws SQLException, NamingException, JMSException {
        final int end = start + num;
        final Connection dbConnection = getConnection();
        try
        {
            // The column ordering here and in the schema (schema_O.sql) must match, so that flat file generation properly maps to columns
            final String insertOrderStmtTxt = "insert into O_ORDERS"
                    + " (O_ID, O_C_ID, O_OL_CNT, O_DISCOUNT, O_TOTAL, O_STATUS, O_ENTRY_DATE, O_SHIP_DATE, O_VERSION)"
                    + " values (?,?,?,?,?,?,?,?,?)";
            final String insertOrderLineStmtTxt = "insert into O_ORDERLINE"
                    + " (OL_ID, OL_O_ID, OL_I_ID, OL_QTY, OL_TOTAL_VALUE, OL_MSRP, OL_STATUS, OL_SHIP_DATE, OL_VERSION)"
                    + " values (?,?,?,?,?,?,?,?,?)";
            final PreparedStatement insertOrderStatement = dbConnection
                  .prepareStatement(insertOrderStmtTxt);
            try {
                final PreparedStatement insertOrderLineStatement = dbConnection
                      .prepareStatement(insertOrderLineStmtTxt);
                try {
                    useBadCreditCustomers(numCustomers, seedGen);
                    try {
                        loadOrders0(start, end, numCustomers, numItems, dbConnection, insertOrderStatement, insertOrderLineStatement);
                    } finally {
                        releaseBadCreditCustomers();
                    }
                } finally {
                    insertOrderLineStatement.close();
                }
            } finally {
                insertOrderStatement.close();
            }
        } finally {
            dbConnection.close();
        }
    }

    private void generateSortedItemIds(final String[] itemIds, final int count, final int numItems) {
        for (int k = 0; k < count; k++) {
            do {
                itemIds[k] = Assembly.getAssemblyId(r.random(1, numItems));
                int l;
                for (l = 0; l < k; l++) {
                    if (itemIds[k].equals(itemIds[l])) {
                        break;
                    }
                }
                if (l == k) {
                    break;
                }
            } while (true);
        }
        Arrays.sort(itemIds, 0, count);
    }

    private void loadOrders0(final int start, final int end, final int numCustomers, final int numItems,
            final Connection dbConnection,
            final PreparedStatement insertOrderStatement,
            final PreparedStatement insertOrderLineStatement) throws SQLException, NamingException, JMSException {
        final PreparedStatement[] preparedStatements = new PreparedStatement[] {
                insertOrderStatement, insertOrderLineStatement};
        final String olIid[] = new String[ORDER_SIZE];
        int batchCount = 0;
        try {
            for (int oId = start; oId < end; oId++) {
                if (infrequentChecker.isStopping()) return;
                int cID;
                do {
                    cID = r.random(1, numCustomers);
                } while (badCreditCustomers.contains(cID));
                final int oCid = cID;
                final int oOlcnt = r.random(1, ORDER_SIZE);
                double oTotal = 0.0;
                final int oStatus = r.random(1, 3);
                double MSRPOrderTotal = 0.0;
                
                generateSortedItemIds(olIid, oOlcnt, numItems);
                
                for (int k = 0; k < oOlcnt; k++) {
                    final int olQty;
                    if (r.random(1, 100) <= 10) {
                        olQty = r.random(1, 99);
                    } else {
                        olQty = r.random(1, 9);
                    }
                    final int olStatus;
                    if (olQty > 20) {
                        olStatus = 3;
                    } else {
                        olStatus = oStatus;
                    }
                    final double olValue = r.drandom(10000.00, 20000.00);
                    final double olMSRP = olValue + r.drandom(5000.00, 20000.00);
                    final double olTotalValue = olValue * olQty;
                    MSRPOrderTotal += olMSRP * olQty;
                    oTotal += olTotalValue;
    
                    insertOrderLineStatement.setInt(1, k + 1);
                    insertOrderLineStatement.setInt(2, oId);
                    insertOrderLineStatement.setString(3, olIid[k]);
                    insertOrderLineStatement.setInt(4, olQty);
                    insertOrderLineStatement.setDouble(5, olTotalValue);
                    insertOrderLineStatement.setDouble(6, olMSRP);
                    insertOrderLineStatement.setInt(7, olStatus);
                    insertOrderLineStatement.setTimestamp(8, olStatus == 3 ? Helper.NOW_TIMESTAMP : null);
                    insertOrderLineStatement.setInt(9, 0);
                    insertOrderLineStatement.addBatch();
                }
                insertOrderLineStatement.executeBatch();
                final double oDiscount = (1.00 - (oTotal / MSRPOrderTotal)) * 100;
                // now create order row
                insertOrderStatement.setInt(1, oId);
                insertOrderStatement.setInt(2, oCid);
                insertOrderStatement.setInt(3, oOlcnt);
                insertOrderStatement.setDouble(4, oDiscount);
                insertOrderStatement.setDouble(5, oTotal);
                insertOrderStatement.setInt(6, oStatus);
                insertOrderStatement.setTimestamp(7, Helper.NOW_TIMESTAMP);
                insertOrderStatement.setDate(8, oStatus == 3 ? Helper.NOW_DATE : null);
                insertOrderStatement.setInt(9, 0);
                insertOrderStatement.addBatch();
                batchCount++;
                batchCount = DatabaseHelper.executeBatchIfFull(dbConnection, batchCount, preparedStatements);
            }
        } finally {
            DatabaseHelper.executeBatch(dbConnection, batchCount, preparedStatements);  // load remaining rows
        }
    }
}
