/*++

  Copyright (c) 1998 Microsoft Corporation   
  
    Module Name:
    spinLock.h
    
      R/W spin lock utility functions. Very optimistic. Very fuzzy.
      
    Created: Neel Jain (njain)

    Warnings:
    - No checking is performed on the parameters.
    - Can spin forever
    - Too many readers can be confused with a writer,
      but that would be the least of your problems
    - Writers always have priority (easily fixed)

    Modified:
    TYHuang: make it compile under VC++. Changed name to Dual.
          
--*/


#ifndef _DUAL_SPINLOCK_HXX_
#define _DUAL_SPINLOCK_HXX_

#include <windows.h>
#include "define.h"
#include "spinlock.hxx"

typedef volatile long DualSpinLock[2];

__inline void initDualSpinLock(DualSpinLock sl)
{
    sl[0] = SL_FREE;
    sl[1] = 0;
}

__inline void acquireDualShared(DualSpinLock sl)
{
    long currentState, writersWaiting;
    
    while (1) {
        
        currentState = sl[0];
        writersWaiting = sl[1];
        
        if ((currentState != SL_LOCKED) && (!writersWaiting)) {
            if (currentState == (long) InterlockedCompareExchangePrivate((long *) &(sl[0]), currentState + 1, currentState)) {
                return;
            }
            else {
                continue;
            }
        }
        
        else {
            SwitchToThread();
        }
        
    }
}

__inline void acquireDualSharedWithSpin(DualSpinLock sl, unsigned __int32 spinCount)
{
    unsigned __int32 i;
    long currentState, writersWaiting;
    
    while (1) {
        
        for (i = 0; i <= spinCount; i++) {
            
            currentState = sl[0];
            writersWaiting = sl[1];
            
            if ((currentState != SL_LOCKED) && (!writersWaiting)) {
                if (currentState == (long) InterlockedCompareExchangePrivate((long *) &(sl[0]), currentState + 1, currentState)) {
                    return;
                }
            }
            
        }
        
        SwitchToThread();
        
    }
}

__inline void releaseDualShared(DualSpinLock sl)
{
    InterlockedDecrement((long *) &(sl[0]));
}

__inline void acquireDualExclusive(DualSpinLock sl)
{
    InterlockedIncrement((long *) &(sl[1]));
    
    if (sl[0] == SL_FREE) {
        if (SL_FREE == InterlockedCompareExchangePrivate((long *) &(sl[0]), SL_LOCKED, SL_FREE)) {
            return;
        }
    }
    
    
    while (1) {
        
        if (sl[0] == SL_FREE) {
            if (SL_FREE == InterlockedCompareExchangePrivate((long *) &(sl[0]), SL_LOCKED, SL_FREE)) {
                return;
            }
            else {
                continue;
            }
        }
        else {
            SwitchToThread();
        }
    }
}

__inline void acquireDualExclusiveWithSpin(DualSpinLock sl, unsigned __int32 spinCount)
{
    unsigned __int32 i;
    
    InterlockedIncrement((long *) &(sl[1]));

    if (sl[0] == SL_FREE) {
        if (SL_FREE == InterlockedCompareExchangePrivate((long *) &(sl[0]), SL_LOCKED, SL_FREE)) {
            return;
        }
    }
        
    while (1) {
        
        for (i = 0; i <= spinCount; i++)
        {
            if (sl[0] == SL_FREE) {
                if (SL_FREE == InterlockedCompareExchangePrivate((long *) &(sl[0]), SL_LOCKED, SL_FREE)) {
                    return;
                }
            }
        }
        
        SwitchToThread();
    }
}


__inline void releaseDualExclusive(DualSpinLock sl)
{
    sl[0] = SL_FREE;
    InterlockedDecrement((long *) &(sl[1]));
}


#endif // _DUAL_SPINLOCK_HXX_
